module net.BurtonRadons.dedit.main;

/* A nascent syntax-hilighting D editor.  More a sample than an example.
 */

import net.BurtonRadons.dig.main;
import std.file;
import std.string;
import std.path;

import net.BurtonRadons.dedit.projectView;
import net.BurtonRadons.dedit.document;
import net.BurtonRadons.dedit.view;
import net.BurtonRadons.dedit.syntaxHighlighter;

const char [] register = "dedit.";

/* A tool that can be executed. */
class ExternalTool
{
    char [] title; /**< Descriptive title. */
    char [] command; /**< Executable path.  This will expand certain special strings.  $(ItemPath) becomes the
        * complete file name of the current document (for example, 'c:/src/foo.d').  $(ItemDir) becomes the path to
        * the current document (for example, 'c:/src/').  $(ItemFileName) becomes the filename of the current 
        * document (for example, 'foo.d').  $(ItemExt) becomes the extension of the current document or null if
        * it has none (for example, '.d').  $(CurLine) becomes the one-based current line of the current document.
        * $(CurCol) becomes the one-based current column of the current document.  $(ProjDir) becomes the directory of the
        * current project (for example, 'c:/src').  $(ProjFileName) becomes the file name of the current project
        * (for example, 'foo.dprj"). */
    char [] initialDirectory = "$(ProjDir)"; /**< Where to start execution.  These have the same expansion capabilities of #command. */
    bit saveAllFiles = true; /**< Save all files in the project before execution. */
    bit useOutputWindow; /**< Not used. */
    bit promptForArguments; /**< Give the user a dialog box filled with the current arguments and allow him to edit or cancel it. */
    bit closeOnExit; /**< Close the console on exit. */
    bit deditSelfDevelopment; /**< Upon successful completion of the task, shutdown dedit and run "make deditRun" asynchronously. */
   
    /** Execute the command. */ 
    void execute (View view)
    {
        alias Control.OS os;
        char [] oldDirectory;
        char [] [] args;
        char [] scom;
        
        if (saveAllFiles)
        {
            view.run ("FileSaveAll");
            global.projectView.paint ();
        }
        
        oldDirectory = os.currentWorkingDirectory ();
        os.currentWorkingDirectory (expand (initialDirectory));
       
        args ~= expand (command); 

        int result;
        
        result = os.system ("%" ~ os.pathDir (os.commandName ()) ~ "deditexec", args);
       
        if (deditSelfDevelopment && !result)
        {
            if (view.projectName !== null)
                view.run ("ProjectSave");
    
            view.registry.saveString (register ~ "project", view.projectName);
            view.confirmModified ();
            args = null;
            args ~= "deditRun";
            os.system ("&make", args);
            global.window.quit ();
        } 
        
        os.currentWorkingDirectory (oldDirectory);
    }
    
    /** Expand special strings. */
    char [] expand (char [] text)
    {
        alias Control.OS os;
        Document document = global.view.document;
        
        text = replace (text, "$(ItemPath)", document.filename);
        text = replace (text, "$(ItemDir)", os.pathDir (document.filename));
        text = replace (text, "$(ItemFileName)", os.pathNotDir (document.filename));
        text = replace (text, "$(ItemExt)", os.pathSuffix (document.filename));
        text = replace (text, "$(CurLine)", fmt ("%d", document.line + 1));
        text = replace (text, "$(CurCol)", fmt ("%d", document.offset + 1));
        text = replace (text, "$(ProjDir)", os.pathDir (global.view.projectName));
        text = replace (text, "$(ProjFileName)", os.pathNotDir (global.view.projectName));
        return text;
    }
    
    class Settings
    {
        ExternalTool tool;
        CheckBox saveAllFilesBox;
        CheckBox promptForArgumentsBox;
        CheckBox closeOnExitBox;
        CheckBox deditSelfDevelopmentBox;
        EditText titleBox;
        EditText commandBox;
        EditText initialDirectoryBox;
        Frame frame;
        
        void saveAllFilesToggle () { saveAllFilesBox.checked (!saveAllFilesBox.checked ()); }
        void promptForArgumentsToggle () { promptForArgumentsBox.checked (!promptForArgumentsBox.checked ()); }
        void closeOnExitToggle () { closeOnExitBox.checked (!closeOnExitBox.checked ()); }
        void deditSelfDevelopmentToggle () { deditSelfDevelopmentBox.checked (!deditSelfDevelopmentBox.checked ()); }
        
        this (ExternalTool tool, View view, char [] caption, bit showTitle)
        {
            char [] captionText = caption; /* Because of the "with" */
            Pane pane;
            GroupBox group;
            
            this.tool = tool;
            
            with (frame = new Frame)
            {
                caption (captionText);
                resizable (false);
                maximizable (false);
            }
            
            with (group = new GroupBox (frame))
                grid (0, 0), caption ("Settings");
            
            int b = showTitle ? 1 : 0;
            
            if (showTitle)
                with (new Label (group)) grid (0, 0), sticky (">"), caption ("Title:");
            with (new Label (group)) grid (0, b), sticky (">"), caption ("Command:");
            with (new Label (group)) grid (0, b + 1), sticky (">"), caption ("Initial Directory:");
                
            with (saveAllFilesBox = new CheckBox (group))
            {
                grid (0, b + 2, 2, 1);
                caption ("Save All Files");
                checked (tool.saveAllFiles);
                onClick.add (&saveAllFilesToggle);                
            }
            
            with (promptForArgumentsBox = new CheckBox (group))
            {
                grid (0, b + 3, 2, 1);
                caption ("Prompt for Arguments");
                checked (tool.promptForArguments);
                onClick.add (&promptForArgumentsToggle);
            }
            
            with (closeOnExitBox = new CheckBox (group))
            {
                grid (0, b + 4, 2, 1);
                caption ("Close on Exit");
                checked (tool.closeOnExit);
                onClick.add (&closeOnExitToggle);
            }
            
            with (deditSelfDevelopmentBox = new CheckBox (group))
            {
                grid (0, b + 5, 2, 1);
                caption ("Dedit Self-development");
                checked (tool.deditSelfDevelopment);
                onClick.add (&deditSelfDevelopmentToggle);
            }
            
            with (pane = new Pane (frame))
            {
                grid (0, 1);
                sticky (">");
                with (new Button (pane))
                {
                    grid (0, 0);
                    caption ("&Okay");
                    onClick.add (&doOkay);
                }
                
                with (new Button (pane))
                {
                    grid (1, 0);
                    caption ("&Cancel");
                    onClick.add (&doCancel);
                }
            }
           
            if (showTitle) with (titleBox = new EditText (group))
            {
                grid (1, 0);
                text (tool.title);
                sticky ("<>");
                width (200);
            }
            
            with (commandBox = new EditText (group))
            {
                grid (1, b);
                text (tool.command);
                sticky ("<>");
                width (200);
            }
            
            with (initialDirectoryBox = new EditText (group))
            {
                grid (1, b + 1);
                sticky ("<>");
                text (tool.initialDirectory);
            }
            
            frame.display ();
        }

        void doOkay (Event e)
        {
            tool.saveAllFiles = saveAllFilesBox.checked ();
            tool.deditSelfDevelopment = deditSelfDevelopmentBox.checked ();
            tool.promptForArguments = promptForArgumentsBox.checked ();
            tool.closeOnExit = closeOnExitBox.checked ();
            if (titleBox)
                tool.title = titleBox.text ();
            tool.command = commandBox.text ();
            tool.initialDirectory = initialDirectoryBox.text ();
            delete frame;
        }
        
        void doCancel (Event e)
        {
            delete frame;
        }        
    }
    
    /** Show a settings dialog for the tool. */
    void settings (View view, char [] caption, bit showTitle)
    {
        new Settings (this, view, caption, showTitle);
    }    
}

/* Parameters for various file types. */
class TabParams
{
    SyntaxHighlighter type; /* Type of file this applies to, null for general. */
    bit useDefault = true; /* Use default parameters. */
    int tabSize = 8; /* Size of tab characters in spaces. */
    int indentSize = 4; /* Size of indentation in spaces. */
    bit keepTabs = false; /* Whether to keep tabs normally or use spaces. */

    /* Expand any indentation. */
    char [] expandIndent (char [] string)
    {
        for (int c; ; c ++)
            if (c >= string.length || !std.ctype.isspace (string [c]))
                return string;
            else if (string [c] == '\t')
            {
                char [] add = new char [tabSize - (c % tabSize)];

                add [] = " ";
                string = string [0 .. c] ~ add ~ string [c + 1 .. string.length];
                c += string.length - 1;
            }
    }

    /* Take the initial whitespace and reduce it as much as possible. */
    char [] reduceIndent (char [] string)
    {
        if (!keepTabs)
            return expandIndent (string);

    outer:
        for (int c; ; c ++)
            if (c >= string.length || !std.ctype.isspace (string [c]))
                return string;
            else if (c + tabSize <= string.length)
            {
                for (int d; d < tabSize; d ++)
                    if (string [c + d] != ' ')
                        continue outer;

                string = string [0 .. c] ~ "\t" ~ string [c + tabSize .. string.length];
                c --;
            }
    }
}

class Global
{
    Color colorComment;
    Color colorIdentifier;
    Color colorKeyword;
    Color colorNumber;
    Color colorSpecialIdentifier;
    Color colorString;
    Color colorSymbol;
    Color colorText;
    Color colorCursor;
    Color colorSelection;
    Color background; /* Background color. */

    ColorSelector colorSelector; /* The main color selector instance. */

    Font font;
    int fontHeight; /* Height of a font line in pixels. */

    Font projectFont;
    int projectFontHeight;

    Color projectBackground;
    Color projectSelection;
    Color projectFile;
    Color projectFileModified;

    Frame window;
    ProjectView projectView;
    View view;
    StatusBar statusBar;

    TabParams [] tabParams; /* List of parameters. */
    
    /* Clear to defaults. */
    void clear ()
    {
        background = AColor (255, 255, 255);

        colorComment = AColor (0, 128, 0);
        colorNumber = AColor (128, 64, 140);
        colorString = AColor (0, 0, 128);
        colorIdentifier = AColor (0, 0, 0);
        colorSpecialIdentifier = AColor (100, 196, 255);
        colorKeyword = AColor (128, 128, 255);
        colorSymbol = AColor (128, 64, 0);
        colorText = AColor (0, 0, 0);
        colorSelection = AColor (255, 196, 196);
        colorCursor = AColor (0, 0, 0);

        font = window.font ();
        font.bold (false);
        font.underline (false);
        font.strikeout (false);
        font.height (15);
        fontHeight = font.height (window);

        projectFile = AColor (0, 0, 0);
        projectFileModified = AColor (128, 0, 0);
        projectBackground = AColor (245, 245, 245);
        projectSelection = AColor (168, 180, 200);
        projectFont = font;
    }

    void love (bit load)
    {
        /* Love values in the registry. */
        with (Control.registry)
        {
            loveColor (register ~ "background", background, load);
            loveColor (register ~ "colorComment", colorComment, load);
            loveColor (register ~ "colorNumber", colorNumber, load);
            loveColor (register ~ "colorString", colorString, load);
            loveColor (register ~ "colorIdentifier", colorIdentifier, load);
            loveColor (register ~ "colorSpecialIdentifier", colorSpecialIdentifier, load);
            loveColor (register ~ "colorKeyword", colorKeyword, load);
            loveColor (register ~ "colorSymbol", colorSymbol, load);
            loveColor (register ~ "colorText", colorText, load);
            loveColor (register ~ "colorCursor", colorCursor, load);
            loveColor (register ~ "colorSelection", colorSelection, load);
            loveFont (register ~ "font", font, load);
            loveFont (register ~ "projectFont", projectFont, load);
            loveColor (register ~ "projectBackground", projectBackground, load);
            loveColor (register ~ "projectSelection", projectSelection, load);
            loveColor (register ~ "projectFile", projectFile, load);
            loveColor (register ~ "projectFileModified", projectFileModified, load);

            if (load)
            {
                int count;

                loadInt (register ~ "tabParamsCount", count);
                tabParams = null;

                for (int c; c < count; c ++)
                {
                    char [] r = fmt ("%.*stabParams%d_", register, c);
                    TabParams p;

                    p = new TabParams ();

                    char [] name;

                    loadString (r ~ "type", name);
                    if (name.length && name !== null)
                    {
                        p.type = SyntaxHighlighter.findName (name);
                        if (p.type === null)
                            continue;
                    }

                    int v;

                    if (loadInt (r ~ "useDefault", v))
                        p.useDefault = (bit) v;
                    if (loadInt (r ~ "tabSize", v))
                        p.tabSize = v;
                    if (loadInt (r ~ "indentSize", v))
                        p.indentSize = v;
                    if (loadInt (r ~ "keepTabs", v))
                        p.keepTabs = (bit) v;

                    tabParams ~= p;
                }
            }
            else
            {
                char [] string;
                TabParams [] list = tabParams;

                saveInt (register ~ "tabParamsCount", list.length);
                for (int c; c < list.length; c ++)
                {
                    char [] r = fmt ("%.*stabParams%d_", register, c);
                    TabParams p = list [c];

                    saveString (r ~ "type", p.type !== null ? p.type.name () : "null");
                    saveInt (r ~ "useDefault", p.useDefault);
                    saveInt (r ~ "tabSize", p.tabSize);
                    saveInt (r ~ "indentSize", p.indentSize);
                    saveInt (r ~ "keepTabs", p.keepTabs);
                }
            }
        }

        if (load)
        {
            fontHeight = font.height (window);
            projectFontHeight = font.height (window);
        }
    }

    void load () { love (true); }
    void save () { love (false); }
    
    /** The document has been renamed. */
    void documentRenamed (Document document)
    {
        ProjectView.DocumentRow row = projectView.findDocument (document);
        
        if (row)
            row.sortChange ();
    }
    
    /** The document is being removed. */
    void documentRemoved (Document document)
    {
        ProjectView.DocumentRow row = projectView.findDocument (document);
        
        if (row)
            row.remove ();
    }
}

Global global;

import net.BurtonRadons.dedit.syntaxHighlighter;

class MenuRunner
{
    char [] command;
    View view;

    this (Dispatcher *d, View view, char [] command)
    {
        this.view = view;
        this.command = command;
        d.add (&eval);
    }

    void eval (Event e)
    {
        global.view.run (command);
    }
}

void Runner (Menu menu, char [] name, View view, char [] command)
{
    char [] options = view.commandOptions [command];

    if (find (options, '.') >= 0)
        name ~= "...";

    if (find (options, '>') >= 0)
    {
        view.run (command);
        menu.popup (name, view.returnMenu);
    }
    else
        new MenuRunner (menu.add (name), view, command);
}

void Runner (Menu menu, char [] name, char [] command)
{
    Runner (menu, name, global.view, command);
}

class MenuCommand
{
    struct Line
    {
        char [] name;
        char [] command;
    }

    Line [] lines;

    void add (char [] name, char [] command)
    {
        Line line;

        line.name = name;
        line.command = command;
        lines ~= line;
    }

    void separator ()
    {
        add ("--", null);
    }

    void run (View view)
    {
        Menu menu = new Menu (view);

        for (int c; c < lines.length; c ++)
        {
            char [] command = lines [c].command;
            char [] name = lines [c].name;

            if (name == "--")
            {
                menu.separator ();
                continue;
            }

            char [] options = View.commandOptions [command];

            if (find (options, '?') >= 0)
            {
                if (view.document.filename === null)
                    name ~= "...";
            }

            if (find (options, '>') >= 0)
            {
                if (find (options, '.') >= 0)
                    name ~= "...";
                view.run (command);
                menu.popup (name, view.returnMenu);
            }
            else
                Runner (menu, name, view, command);
        }

        menu.exec ();
    }
}

class StatusBar : Canvas
{
    char [] displayText;

    this (Control parent)
    {
        super (parent);
        onPaint.add (&doPaint);

        int w, h;
        drawboxBorder (w, h, DB.ThinSunken);
        font ().height (font ().height (this) * 0.9);
        height (font ().height (this) + h + 2);
    }

    void display (char [] text)
    {
        displayText = text;
        paint ();
    }
    
    int visibleLine;
    int visibleOffset;
    
    /** Paint if any status displayed has changed. */
    void paintCheck ()
    {
        if (visibleLine != global.view.document.line
         || visibleOffset != global.view.document.offset)
            paint ();
    }

    void doPaint ()
    {
        beginPaint ();

        statusbarBackgroundDraw (0, 0, width (), height ());

        char [] text;
        Document document = global.view.document;
        int dbw, dbh;
        int x = width (), w;

        drawboxBorder (dbw, dbh, DB.ThinSunken);

        textPrint (7, 1 + dbh, displayText);

        visibleLine = document.line;
        visibleOffset = document.offset;
        text = fmt ("Ln %d, Col %d", document.line + 1, document.offset + 1);
        w = 90;
        statusbarPaneDraw (x - w - dbw * 2, 0, x, height ());
        textPrint (x - w - dbw, 1 + dbh, text);

        endPaint ();
    }
}

class MatchList : Frame
{
    class Match
    {
        Document document;
        int line; /* Zero-based line index. */
        char [] text; /* Descriptive text. */
        int mark;
        int offset;

        int opCmp (Object o)
        {
            if (!cast (Match) o)
                return 0;
            Match m = cast (Match) o;
            
            if (document.opCmp (m.document))
                return document.opCmp (m.document);
            return m.line - line;
        }
    }

    class MatchDocument
    {
        Document document;
        Match [] matches;
    }

    MatchDocument [] documents;
    Match [] ordered; /* Ordered in incidence. */

    /* Add an entry into the list. */
    void add (Document document, int line, char [] text, int mark, int offset)
    {
        Match match = new Match ();

        match.document = document;
        match.line = line;
        match.text = text;
        match.mark = mark;
        match.offset = offset;
        ordered ~= match;

        for (int c; c < documents.length; c ++)
            if (documents [c].document === document)
            {
                documents [c].matches ~= match;
                return;
            }

        MatchDocument d = new MatchDocument ();

        d.document = document;
        d.matches ~= match;
        documents ~= d;
        recalculate ();
    }

    void removeMatch (Match m)
    {
        for (int c; c < ordered.length; c ++)
            if (m === ordered [c])
            {
                for ( ; c < ordered.length - 1; c ++)
                    ordered [c] = ordered [c + 1];
                ordered = ordered [0 .. ordered.length - 1];
                break;
            }
    }

    /* Get the entry from the document's selection or the caret. */
    void add (Document document)
    {
        add (document, document.line, document.lines [document.line].dup, document.offset, document.offset);
    }

    void display ()
    {
        create ();
        recalculate ();
        super.display ();
    }

    private bit fCreated;
    int baseWidth;
    List list;

    void create ()
    {
        if (fCreated)
            return;
        fCreated = true;
        caption ("Match List");
        border (0, 0);
        onResized.add (&doResized);
        onMaximized.add (&doResized);
        onMouseWheel.add (&doMouseWheel);

        with (list = new List (this))
        {
            pad (0, 0);
            grid (0, 0);
            widthAndHeight (400, 400);
        }
    }

    void doResized (Event e)
    {
        list.widthAndHeight (e.x, e.y);
        display ();
    }

    void doMouseWheel (Event e)
    {
        list.scroll -= e.wheel * 4;
        list.paint ();
    }

    void recalculate ()
    {
        if (!fCreated)
            return;

        caption (fmt ("Match List (%d matches)", ordered.length));
        list.next = 0;
        for (int c; c < ordered.length; c ++)
        {
            Match m = ordered [c];

            list.next = imax (list.next, list.textWidth (fmt ("%.*s, line %d:", m.document.name (), m.line + 1)));
        }

        list.paint ();
    }

    class List : Canvas
    {
        MatchList matches;
        int scroll = 0;
        int next;
        Match [] selected;
        Match current;
        int mouseStart;

        this (MatchList parent)
        {
            this.matches = parent;
            super (parent);
            vscroll (true);
            onPaint.add (&doPaint);
            onVScroll.add (&doScroll);
            onLButtonDown.add (&doLButtonDown);
            onMouseMove.add (&doMouseMove);
            onLButtonUp.add (&doLButtonUp);

            bind ("Delete", &keyDelete);
            bind ("Down", &keyDown);
            bind ("End", &keyEnd);
            bind ("Home", &keyHome);
            bind ("PageDown", &keyPageDown);
            bind ("PageUp", &keyPageUp);
            bind ("Up", &keyUp);

            bind ("Shift-Down", &keyShiftDown);
            bind ("Shift-End", &keyShiftEnd);
            bind ("Shift-Home", &keyShiftHome);
            bind ("Shift-PageDown", &keyShiftPageDown);
            bind ("Shift-PageUp", &keyShiftPageUp);
            bind ("Shift-Up", &keyShiftUp);
        }

        void currentIndex (int index)
        {
            index = imid (0, index, matches.ordered.length - 1);
            if (matches.ordered.length == 0)
                current = null;
            else
                current = matches.ordered [index];

            int th = textHeight ();

            if (index > scroll + height () / th - 4)
                scroll = index - height () / th + 4;
            if (index < scroll + 3)
                scroll = index - 3;
        }

        int currentIndex ()
        {
            return orderedIndex (current);
        }

        void keyDelete ()
        {
            Match [] s = selected;
            int ci = orderedIndex (current);

            if (s.length == 0)
            {
                if (!current)
                    return;
                matches.removeMatch (current);
            }
            else
            {
                for (int c; c < s.length; c ++)
                {
                    int di = orderedIndex (s [c]);

                    if (di < ci)
                        ci --;
                    matches.removeMatch (s [c]);
                }
            }

            if (ci == matches.ordered.length)
                ci --;
            selected = null;
            if (ci < 0)
                current = null;
            else
                current = matches.ordered [ci];
            matches.recalculate ();
        }

        void keyDown ()
        {
            if (!current)
                return;
            currentIndex (currentIndex () + 1);
            mouseStart = currentIndex ();
            selected = null;
            paint ();
        }

        void keyShiftDown ()
        {
            if (!current)
                return;
            currentIndex (currentIndex () + 1);
            selectFrom (mouseStart, currentIndex ());
            paint ();
        }

        void keyEnd ()
        {
            currentIndex (matches.ordered.length - 1);
            mouseStart = currentIndex ();
            selected = null;
            paint ();
        }

        void keyShiftEnd ()
        {
            currentIndex (matches.ordered.length - 1);
            selectFrom (mouseStart, currentIndex ());
            paint ();
        }

        void keyHome ()
        {
            selected = null;
            currentIndex (0);
            mouseStart = currentIndex ();
            paint ();
        }

        void keyShiftHome ()
        {
            currentIndex (0);
            selectFrom (mouseStart, currentIndex ());
            paint ();
        }

        void keyPageDown ()
        {
            if (!current)
                return;
            currentIndex (currentIndex () + imax (1, height () / textHeight () - 4));
            mouseStart = currentIndex ();
            selected = null;
            paint ();
        }

        void keyShiftPageDown ()
        {
            if (!current)
                return;
            currentIndex (currentIndex () + imax (1, height () / textHeight () - 4));
            selectFrom (mouseStart, currentIndex ());
            paint ();
        }
        
        void keyPageUp ()
        {
            if (!current)
                return;
            currentIndex (currentIndex () - imax (1, height () / textHeight () - 4));
            mouseStart = currentIndex ();
            selected = null;
            paint ();
        }

        void keyShiftPageUp ()
        {
            if (!current)
                return;
            currentIndex (currentIndex () - imax (1, height () / textHeight () - 4));
            selectFrom (mouseStart, currentIndex ());
            paint ();
        }
        
        void keyUp ()
        {
            if (!current)
                return;
            currentIndex (currentIndex () - 1);
            mouseStart = currentIndex ();
            selected = null;
            paint ();
        }

        void keyShiftUp ()
        {
            if (!current)
                return;
            currentIndex (currentIndex () - 1);
            selectFrom (mouseStart, currentIndex ());
            paint ();
        }

        void doScroll (Event e)
        {
            Event.Scroll type = e.scrollType;
            int point = e.scrollPoint;
            int h = height ();
            int th = textHeight ();

            switch (type)
            {
                case e.Scroll.Top: scroll = 0; break;
                case e.Scroll.Bottom: scroll = matches.ordered.length - h / th; break;
                case e.Scroll.LineUp: scroll --; break;
                case e.Scroll.LineDown: scroll ++; break;
                case e.Scroll.PageDown: scroll -= imax (1, h / th - 4); break;
                case e.Scroll.PageUp: scroll += imax (1, h / th - 4); break;
                case e.Scroll.Drop:
                case e.Scroll.Track:
                    scroll = point;

                case e.Scroll.End:
                    break;
            }

            paint ();
        }

        void addSelection (Match m)
        {
            if (selectionIndex (m) >= 0)
                return;
            selected ~= m;
        }

        int selectionIndex (Match m)
        {
            for (int c; c < selected.length; c ++)
                if (selected [c] === m)
                    return c;

            return -1;
        }

        int orderedIndex (Match m)
        {
            for (int c; c < matches.ordered.length; c ++)
                if (matches.ordered [c] === m)
                    return c;

            return -1;
        }

        void selectFrom (int start, int end)
        {
            start = imid (0, start, matches.ordered.length - 1);
            end = imid (0, end, matches.ordered.length - 1);
            if (start < end)
                selected = matches.ordered [start .. end + 1].dup;
            else
                selected = matches.ordered [end .. start + 1].dup;
        }

        void doLButtonDown (Event e)
        {
            int i = e.y / textHeight () + scroll;

            if (i < 0 || i >= matches.ordered.length)
                return;
            Match m = matches.ordered [i];

            if (e.control)
            {
                for (int c; ; c ++)
                    if (c >= selected.length)
                    {
                        selected ~= m;
                        break;
                    }
                    else if (selected [c] === m)
                    {
                        for ( ; c < selected.length - 1; c ++)
                            selected [c] = selected [c + 1];
                        selected.length = selected.length - 1;
                        break;
                    }
            }
            else if (e.shift && current)
            {
                int st = i, en = orderedIndex (current);

                if (st > en)
                {
                    st = en;
                    en = i;
                }

                for (int c = st; c <= en; c ++)
                    addSelection (matches.ordered [c]);
            }
            else
            {
                selected = null;
                mouseStart = i;
                captureMouse ();
                selectFrom (i, i);
            }

            current = m;
            m.document.line = m.line;
            m.document.offset = m.offset;
            global.view.switchDocument (m.document);
            global.view.caretChange ();
            paint ();
        }

        void doLButtonUp (Event e)
        {
            if (!isCaptor ())
                return;
            releaseMouse ();

            int i = imid (0, e.y / textHeight () + scroll, matches.ordered.length - 1);

            selectFrom (mouseStart, i);
            current = matches.ordered [i];
            paint ();
        }

        void doMouseMove (Event e)
        {
            if (!isCaptor ())
                return;

            int i = imid (0, e.y / textHeight () + scroll, matches.ordered.length - 1);

            selectFrom (mouseStart, i);
            current = matches.ordered [i];
            paint ();
        }

        void doPaint ()
        {
            beginPaint ();
            clear (Color.White);

            int th = textHeight ();
            Document previous;
            int start;

            scroll = imin (scroll, matches.ordered.length - height () / th);
            scroll = imax (0, scroll);

            vscrollRange (0, matches.ordered.length);
            vscrollPoint (scroll);
            vscrollPage (height () / th);

            for (int c = scroll; c < matches.ordered.length; c ++)
            {
                int y = th * (c - scroll);
                Match m;

                if (c < 0)
                    continue;
                m = matches.ordered [c];
                if (y > height ())
                    break;

                for (int i; i < selected.length; i ++)
                    if (m === selected [i])
                    {
                        penClear ();
                        brushColor (AColor (196, 196, 255));
                        rect (0, y, visualWidth () + 1, y + th + 1);
                        break;
                    }

                if (m === current)
                    drawbox (0, y, visualWidth (), y + th, DB.Focus, false);

                if (previous !== m.document)
                {
                    textFormat (3, y, "%.*s, line %d:", m.document.name (), m.line + 1);
                    previous = m.document;
                    start = textWidth (m.document.name () ~ ", line ");
                }
                else
                    textFormat (3 + start, y, "%d:", m.line + 1);

                int x = next + 6;
                textPrintEllipses (x, y, width () - x - 17, m.mark, m.text);
            }

            endPaint ();
        }
    }
}

class Program : Frame
{
    void prepareMenu (Menu menu)
    {
        Menu sub;

        with (sub = new Menu (this))
        {
            menu.popup ("&File", sub);
            Runner (sub, "&New", "FileNew");
            Runner (sub, "&Open", "FileOpen");
            Runner (sub, "&Close", "FileClose");
            Runner (sub, "&Save", "FileSave");
            Runner (sub, "Save A&ll", "FileSaveAll");
            separator ();
            Runner (sub, "E&xit", "Exit");
        }

        with (sub = new Menu (this))
        {
            menu.popup ("&Edit", sub);
            Runner (sub, "&Undo", "Undo");
            Runner (sub, "&Redo", "Redo");
            separator ();
            Runner (sub, "Cu&t", "Cut");
            Runner (sub, "&Copy", "Copy");
            Runner (sub, "&Paste", "Paste");
            Runner (sub, "&Delete", "Delete");
            separator ();
            Runner (sub, "Select Al&l", "SelectAll");
            separator ();
            Runner (sub, "&Find", "Find");
            Runner (sub, "Find &Next", "FindNext");
        }

        with (sub = new Menu (this))
        {
            menu.popup ("&Project", sub);
            Runner (sub, "&New", "ProjectNew");
            Runner (sub, "&Open", "ProjectOpen");
            Runner (sub, "&Save", "ProjectSave");
            Runner (sub, "S&tatistics", "ProjectStats");
            Runner (sub, "Compile Settings", "ProjectCompileSettings");
            Runner (sub, "Run Settings", "ProjectRunSettings");
        }

        with (sub = new Menu (this))
        {
            menu.popup ("&Tools", sub);
            Runner (sub, "&Options", "Options");
            Runner (sub, "&Keyboard Map", "KeyboardMap");
        }
    }

    this ()
    {
        MenuBar menu = new MenuBar ();

        super (menu);
        caption ("D Editor");
        border (0, 0);
        pad (0, 0);

        prepareMenu (menu);

        with (global.projectView = new ProjectView (this))
        {
            grid (0, 0);
            pad (0, 0);
            sticky ("<>^v");
            vscroll (true);
            suggestWidth (200);
            orderedColumn (0).width = width ();
        }

        with (global.view = new View (this))
        {
            suggestWidthAndHeight (400, 400);
            sticky ("<>^v");
            grid (1, 0);
            pad (0, 0);
            hscroll (false);
            vscroll (true);
        }

        with (global.statusBar = new StatusBar (this))
        {
            grid (0, 1, 2, 1);
            sticky ("<>");
            pad (0, 0);
        }

        onResized.add (&resize);
        onMaximized.add (&resize);
        onMouseWheel.add (&doMouseWheel);
        onClose.add (&doClose);
        onActive.add (&doActive);
    }

    void doActive (Event e)
    {
        global.view.makeFocus ();
    }

    void doClose (Event e)
    {
        if (global.view.projectName !== null)
            global.view.run ("ProjectSave");

        registry.saveString (register ~ "project", global.view.projectName);
        if (global.view.confirmModified ())
            quit ();
    }

    void doMouseWheel (Event e)
    {
        global.view.onMouseWheel.notify (e);
    }

    void resize (Event e)
    {
        static bit recurse = false;

        if (recurse)
            return;
        recurse = true;

        int cx = e.x - width ();
        int cy = e.y - height ();

        global.view.width (global.view.width () + cx);
        global.view.height (global.view.height () + cy);
        global.projectView.height (global.projectView.height () + cy);
        global.statusBar.width (e.x);

        display ();

        recurse = false;
    }
}

int main (char [] [] argv)
{
    Program ded;

    global = new Global ();
    ded = new Program ();
    
    try
    {
        global.window = ded;
        global.colorSelector = new ColorSelector ();
        global.clear ();
        global.load ();

        if (argv.length > 1)
        {
            if (argv.length == 2 && getExt (argv [1]) == "dprj")
            {
                global.view.projectName = argv [1];
                global.view.run ("ProjectReload");
            }
            else
            {
                global.projectView.restart ();
                global.view.projectName = null;
                for (int c = 1; c < argv.length; c ++)
                    global.view.loadFile (argv [c]);
            }
        }
        else if (global.view.projectName !== null)
            global.view.run ("ProjectReload");
        else
        {
            global.projectView.restart ();
            global.view.newDocument ();
        }

        ded.showModal ();
    }
    catch (Object obj)
    {
        Document [] documents;
        Control.MB mask = Control.MB.IconError;
        char [] t;

        if (ded === null)
            throw obj;
        if (ded !== null && global.view !== null)
            documents = global.view.documents;

        t ~= "There has been an exception of type " ~ obj.classinfo.name ~ ":\n";
        t ~= obj.toString () ~ "\n";

        char [] [] names;

        for (int c; c < documents.length; c ++)
            if (documents [c].modified)
                names ~= documents [c].name ();

        if (names.length)
        {
            char [] text;

            mask |= Control.MB.YesNo;
            t ~= "\n\nThe following files have been modified. ";
            t ~= "Because their contents may be corrupted, I can save them to "
                 "their filename with a '~' appended to the end.  Would you "
                 "like me to do this?  The files are:\n\n";

            for (int c; c < names.length; c ++)
            {
                t ~= names [c];
                if (c < names.length - 1)
                    t ~= "\n\t";
            }
        }

        char [] r = ded.messageBox ("Exception!", t, mask);

        if (r == "Yes")
        {
            for (int c; c < documents.length; c ++)
                if (documents [c].modified && documents [c].filename !== null)
                    documents [c].saveTo (documents [c].filename ~ "~");
        }

        throw obj;
    }
    
    return 0;
}
